/****************************************************************************
 *   $Id:: main.c 2010-11-09 02:23:44Z nxp12832		                 		$
 *   Project: NXP Power Profiles Usage Example
 *
 *   Description:
 *     This file contains PLL setting demo using the boot-ROM API calls.
 *
 ****************************************************************************
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * products. This software is supplied "AS IS" without any warranties.
 * NXP Semiconductors assumes no responsibility or liability for the
 * use of the software, conveys no license or title under any patent,
 * copyright, or mask work right to the product. NXP Semiconductors
 * reserves the right to make changes in the software without
 * notification. NXP Semiconductors also make no representation or
 * warranty that such application will be suitable for the specified
 * use without further testing or modification.
****************************************************************************/

#include "LPC11xx.h"
#include "power_api.h"
#include "power_profiles.h"
#include "uart.h"
#include <stdio.h>
#include "divfuncs.h"

/*	Configuration options */
#define UART_BAUD		57600
#define ENABLE_CLKOUT 	0
#define SYS_OSC  		1
#define ITERATION_COUNT	5000000

#define FREQ			50000000UL
//#define FREQ			36000000UL
//#define FREQ			30000000UL
//#define FREQ			12000000UL
//#define FREQ			6000000UL

#define SYS_FREQ		FREQ
/*************************************************************/

/*	Variable set by SystemInit() and used by the UART driver */
extern uint32_t SystemAHBFrequency;
extern volatile uint32_t UARTCount;
extern volatile uint8_t  UARTBuffer[];
/*************************************************************/

/*	Function used by Retarget.c for printf statements */
int  schar(int ch);
int32_t  gchar(void);
/*************************************************************/

/*	Function prototypes */
POWER_PROFILE profile;
uint32_t run_dummy_routine(uint32_t iterations);
volatile uint8_t *string;
void delay_ms(uint32_t time);
void config_low_current_ios(void);
void config_enable_ios(void);	//Enable LED output, UART outputs, Reset, and SW I/O
/*************************************************************/

/*	Menu options */
void print_menu(void){
	printf("\x1b#8");	//VT100 Screen align
	printf("\x1b[2J");	//VT100 Clear screen
#if SYS_OSC
	printf("set_power Power Profiles Demo LPC1100 - MAIN OSC\n\r");
#else
	printf("set_power Power Profiles Demo LPC1100 - IRC OSC\n\r");
#endif /* SYS_OSC */
	printf("Press any digit from '0' - '3' to run a set_power test\n\r");
	printf("    '0' - DEFAULT MODE\n\r"
	       "    '1' - LOW CURRENT MODE\n\r"
	       "    '2' - EFFICIENCY MODE\n\r"
		   "    '3' - PERFORMANCE MODE\n\r");
}
/*************************************************************/



int main(void){
	uint32_t counter = 0;
	volatile static int32_t cmd;

	/*	Enable AHB/APB bridge; ROM; RAM; FLASHREG; FLASHARRAY; GPIO; CT32B0&1; IOCON */
	LPC_SYSCON->SYSAHBCLKCTRL = (1<<0)|(1<<1)|(1<<2)|(1<<3)|(1<<4)|(1<<6)|(1<<9)|(1<<10)|(1<<16);

#if SYS_OSC	
	/*	Enable system oscillator and go into a little dummy delay */
	LPC_SYSCON->SYSOSCCTRL = 0x00;						//No Bypass; 1-20Mhz Range
	LPC_SYSCON->PDRUNCFG &= ~(0x1<<5);					//Enable system oscillator
	for ( counter = 0; counter < 0x100; counter++ );	//Dummy wait for the system osc

	/*	Select the system oscillator as sys_pllclkin	*/
	LPC_SYSCON->SYSPLLCLKSEL = 0x1;
	LPC_SYSCON->SYSPLLCLKUEN = 0;
	LPC_SYSCON->SYSPLLCLKUEN = 1;
	while ( !(LPC_SYSCON->SYSPLLCLKUEN & 0x01) ); /* Wait until updated */
#endif /* SYS_OSC */


#if ENABLE_CLKOUT
	/* Output the Clk onto the CLKOUT Pin PIO0_1 to monitor the freq on a scope */
	LPC_IOCON->PIO0_1	= (1<<0);
	/* Select the MAIN clock as the clock out selection since it's driving the core */
	LPC_SYSCON->CLKOUTCLKSEL = 3;
	/* Set CLKOUTDIV to 6 */
	LPC_SYSCON->CLKOUTDIV = 1;		//	CLKOUT Divider = 10
	/* Enable CLKOUT */
	LPC_SYSCON->CLKOUTUEN = 0;
	LPC_SYSCON->CLKOUTUEN = 1;
	while (!(LPC_SYSCON->CLKOUTUEN & 0x01));
#endif /* ENABLE_CLKOUT */
	
	profile.new_system_clock = 	SYS_FREQ;
	profile.current_system_clock = 12000000UL;
	profile.power_mode = PARAM_DEFAULT;

	if(config_pll_power(&profile)){
		SystemInit();
		UARTInit(UART_BAUD);
		printf("ROM PLL/PWR Config Fail\n\r");
		while(1);			
	}else{
		SystemAHBFrequency = profile.current_system_clock * LPC_SYSCON->SYSAHBCLKDIV;
		UARTInit(UART_BAUD);
	}

#if SYS_OSC
	/*	Turn off the IRC, BOD, ADC & WDT */
	LPC_SYSCON->PDRUNCFG |= (1<<0)|(1<<1)|(1<<3)|(1<<4)|(1<<6);
#else
	/*	Turn off the      BOD, ADC & WDT */
	LPC_SYSCON->PDRUNCFG |=               (1<<3)|(1<<4)|(1<<6);
#endif /* SYS_OSC */

	/*	Init a 32-bit Timer */
	LPC_TMR32B0->TCR = (1<<1);		//Reset and Disable Timer
	LPC_TMR32B0->PR	 = 0;			//Increment at every clock cycle
	
	print_menu();

	config_low_current_ios();
	config_enable_ios();
	
	while(1){
		/*	Check if something was received on the UART */
		if(UARTCount){
			
			cmd = gchar();

			switch(cmd){
				case '0':
					profile.power_mode = PARAM_DEFAULT;
					string	= "DEFAULT MODE";
					break;
				case '1':
					profile.power_mode = PARAM_LOW_CURRENT;
					string	= "LOW CURRENT MODE";
					break;
				case '2':
					profile.power_mode = PARAM_EFFICIENCY;
					string	= "EFFICIENCY MODE";
					break;
				case '3':
					profile.power_mode = PARAM_CPU_EXEC;
					string	= "PERFORMANCE MODE";
					break;
				default:
					break;
			}

			/*	Apply the new frequency via the power profile API call */
			if(config_pll_power(&profile)){
				SystemAHBFrequency = profile.current_system_clock * LPC_SYSCON->SYSAHBCLKDIV;
				UARTInit(UART_BAUD);
				printf("ROM PLL/PWR Config Fail\n\rTry Again\n\r");			
			}else{
				SystemAHBFrequency = profile.current_system_clock * LPC_SYSCON->SYSAHBCLKDIV;
				UARTInit(UART_BAUD);
				delay_ms(20);
			}

			printf("\n\rRunning \x1b[5m%s\x1b[0m Test @ %d Hz\n\r", string, profile.current_system_clock);
			delay_ms(10);
			NVIC_DisableIRQ(UART_IRQn);	//Disable UART IRQ to prevent any unwanted interrupts from the UART UI
			config_low_current_ios();	//Configure IOs to reduce current consumption
			/*	Start Timer */
			LPC_TMR32B0->TCR = (1<<1);	//Reset and Disable Timer
			LPC_TMR32B0->TCR = (1<<0);	//Enable and Start Timer

			/*	Perform process */
			run_dummy_routine(ITERATION_COUNT);
			
			/*	Stop Timer */
			LPC_TMR32B0->TCR = 0;		//STOP Timer
			config_enable_ios();
			NVIC_EnableIRQ(UART_IRQn);	//Enable UART IRQ to allow interrupts from the UART UI
			printf("%s Test Duration:\t%d us\n\r", string, (LPC_TMR32B0->TC / (profile.current_system_clock / 1000000)) );
			printf("\a");				//Generate an bell alert on the VT100 (Tera Term Pro) terminal
			
		}

		/*	uC is alive indicator */
		if(counter == 0x10000){
			counter = 0;
			LPC_GPIO2->DATA ^= (1<<0);	//Toggle LED
		}else
			counter++;

		
	}
}
void delay_ms(uint32_t time){
	/* setup timer #1 for delay */
    LPC_TMR32B1->TCR = 0x02;		/* reset timer */
    LPC_TMR32B1->PR  = 0x00;		/* set prescaler to zero */
    LPC_TMR32B1->MR0 = (time * (profile.current_system_clock / 1000)) - 1;
    LPC_TMR32B1->MCR = 0x04;		/* stop timer on match */
    LPC_TMR32B1->TCR = 0x01;		/* start timer */
  
    /* wait until delay time has elapsed */
    while (LPC_TMR32B1->TCR & 0x01);
	return;
}


uint32_t run_dummy_routine(uint32_t iterations){
	uint32_t i, sum = 0;
	for(i =  0; i < iterations; i++){
		sum += udiv(iterations,13);
	}
	return sum;
}

/*	Function used for print statements */
int  schar(int ch){
	uint8_t chr = ch;
	UARTSend(&chr, 1);
	return 0;
}

/*	Function returns 1st character from the received UARTBuffer */
int32_t gchar(void) {
	uint8_t temp;

	while ( UARTCount == 0 );
	
	LPC_UART->IER = IER_THRE | IER_RLS;			/* Disable RBR */
	  //UARTSend( (uint8_t *)UARTBuffer, UARTCount );
	temp = UARTBuffer[0];
	UARTCount = 0;
	LPC_UART->IER = IER_THRE | IER_RLS | IER_RBR;	/* Re-enable RBR */
	return (int32_t)temp;
}

void config_low_current_ios(void){

	/* Configure all IOs as GPIO w/o pull-up & pull-down resistors */
	LPC_IOCON->PIO2_6			= 0xC0;
	LPC_IOCON->PIO2_0			= 0xC0;
	LPC_IOCON->RESET_PIO0_0 	= 0xC1;      // RST_BOOT/P0_0

	LPC_IOCON->PIO0_1			= 0xC0;      // CLKOUT/CT32B0_MAT2/USB_SOF

	LPC_IOCON->PIO1_8			= 0xC0;
	LPC_IOCON->PIO0_2			= 0xC0;      // SSEL/CT16B0_CAP0	//Trigger input

	LPC_IOCON->PIO2_7			= 0xC0;
	LPC_IOCON->PIO2_8			= 0xC0;
	LPC_IOCON->PIO2_1			= 0xC0;
	LPC_IOCON->PIO0_3			= 0xC0;      // USB VBUS
	LPC_IOCON->PIO0_4			= 0xC0;      // I2C_SCL, no pull-up, inactive
	LPC_IOCON->PIO0_5			= 0xC0;      // I2C_SDA, no pull-up, inactive
	LPC_IOCON->PIO1_9			= 0xC0;      // CT16B1_MAT0
	LPC_IOCON->PIO3_4			= 0xC0;

	LPC_IOCON->PIO2_4			= 0xC0;
	LPC_IOCON->PIO2_5			= 0xC0;
	LPC_IOCON->PIO3_5			= 0xC0;
	LPC_IOCON->PIO0_6			= 0xC0;      // USB SoftConnect
	LPC_IOCON->PIO0_7			= 0xC0;
	LPC_IOCON->PIO2_9			= 0xC0;
	LPC_IOCON->PIO2_10			= 0xC0;
	LPC_IOCON->PIO2_2			= 0xC0;

	LPC_IOCON->PIO0_8			= 0xC0;		// SSP_MISO/CT16B0_MAT0/TRACE_CLK
	LPC_IOCON->PIO0_9			= 0xC0;		// SSP_MOSI/CT16B0_MAT1/TRACE_SWV
	LPC_IOCON->SWCLK_PIO0_10	= 0xC1;		// ARM_SWCLK/P0_10/SSP_CLK/CT16B0_MAT2
	LPC_IOCON->PIO1_10			= 0xC0;		// ADCIN6/CT16B1_MAT1
	LPC_IOCON->PIO2_11			= 0xC0;
	LPC_IOCON->R_PIO0_11		= 0xC1;		// R/P0_11/ADCIN0/CT32B0_MAT3
	LPC_IOCON->R_PIO1_0			= 0xC1;		// R/P1_0/ADCIN1/CT32B1_CAP0
	LPC_IOCON->R_PIO1_1			= 0xC1;		// R/P1_1/ADCIN2/CT32B1_MAT0

	LPC_IOCON->R_PIO1_2			= 0xC1;		// R/P1_2/ADCIN3/CT32B1_MAT1
	LPC_IOCON->PIO3_0			= 0xC0;
	LPC_IOCON->PIO3_1			= 0xC0;
	LPC_IOCON->PIO2_3			= 0xC0;
	LPC_IOCON->SWDIO_PIO1_3		= 0xC1;		// ARM_SWD/P1_3/ADCIN4/CT32B1_MAT2
	LPC_IOCON->PIO1_4			= 0xC0;		// ADCIN5/CT32B1_MAT3/WAKEUP
	LPC_IOCON->PIO1_11			= 0xC0;		// ADCIN7
	LPC_IOCON->PIO3_2			= 0xC0;

	LPC_IOCON->PIO1_5			= 0xC0;      // UART_DIR/CT32B0_CAP0
	LPC_IOCON->PIO1_6			= 0xC0;      // UART_RXD/CT32B0_MAT0
	LPC_IOCON->PIO1_7			= 0xC0;      // UART_TXD/CT32B0_MAT1
	LPC_IOCON->PIO3_3			= 0xC0;

	/* GPIOs at outputs */
	//LPC_GPIO0->DIR = ~(1<<2);	//Trigger Pin input only
	//LPC_GPIO1->DIR = ~(1<<4);	//WAKEUP Pin input only	
	//PIO1_6/RXD is configured as input since the transceiver pin is output.
	//if PIO1_6/RXD is configured as output, driven low and the transceiver is driving high then extra current
	//LPC_GPIO1->DIR = ~((1<<4)|(1<<6));	//WAKEUP Pin and PIO1_6/RXD input only

	LPC_GPIO0->DIR = 0xFFFFFFFF;
	//LPC_GPIO1->DIR = 0xFFFFFFFF;
	LPC_GPIO1->DIR = ~(1<<6);
	LPC_GPIO2->DIR = 0xFFFFFFFF;
	LPC_GPIO3->DIR = 0xFFFFFFFF;


	/* GPIO outputs to LOW */
	LPC_GPIO0->DATA  = 0;
	//LPC_GPIO1->DATA  = (1<<6);	//Drive UART Rx output HIGH to avoid additional current consumption
	LPC_GPIO1->DATA  = 0;
	LPC_GPIO2->DATA  = 0;
	LPC_GPIO3->DATA  = 0;

	#if 1
		LPC_GPIO0->DATA |= (1<<0) | (1<<1);
		LPC_GPIO1->DATA |= (1<<4) | (1<<5);
	#endif

	return;
}

void config_enable_ios(void){
	/*	LED enable */
	LPC_IOCON->PIO2_0 			= 0xD0;	//Switch to default state: GPIO w/ Pull-up resistor enabled

	/*	UART enable */
	LPC_IOCON->PIO1_6			= 0xD1;	//Switch to UART Receive w/ Pull-up resistor enabled
	LPC_IOCON->PIO1_7			= 0xD1;	//Switch to UART Transmit w/ Pull-up resistor enabled

#if !ENABLE_CLKOUT
	LPC_IOCON->PIO0_1			= 0xD1;      // CLKOUT/CT32B0_MAT2/USB_SOF
#endif /* ENABLE_CLKOUT */

	/*	Reset enable */
		//LPC_GPIO0->DIR				&= ~(1<<0);	//Make GPIO0_0 as an input
	LPC_GPIO0->DATA 			|= (1<<0);	//Drive the GPIO0_0 HIGH
		//LPC_IOCON->RESET_PIO0_0 	|= (0x3<<3);//Enable Pull-up resistor
	delay_ms(1);							//Wait for the line to go HIGH
	LPC_IOCON->RESET_PIO0_0 	&= ~(1<<0);	//Switch to Reset input

	/*	Debugger enable */
	LPC_IOCON->SWCLK_PIO0_10	= 0xD0;		// ARM_SWCLK/P0_10/SSP_CLK/CT16B0_MAT2
	LPC_IOCON->SWDIO_PIO1_3		= 0xD0;		// ARM_SWD/P1_3/ADCIN4/CT32B1_MAT2
	return;
}


